//=============================================================================
//
// WarningLight.cpp :
//
// This example shows how a lamp on O8 is set to blinking in dependence of
// the value of an ultrasonic sensor on I1. The lamp blinks in different
// frequencies dependent on the distance of an obstacle (the nearer, the
// faster the blink frequency).
//
//-----------------------------------------------------------------------------
// Disclaimer - Exclusion of Liability
//
// This software is distributed in the hope that it will be useful,but WITHOUT 
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
// FITNESS FOR A PARTICULAR PURPOSE. It can be used an modified by anyone
// free of any license obligations or authoring rights.
//=============================================================================

#include "StdAfx.h"

enum MotorIdx {MOTOR_1 = 0, MOTOR_2, MOTOR_3, MOTOR_4};
enum PwmOut   {OUT1 = 0, OUT2, OUT3, OUT4, OUT5, OUT6, OUT7, OUT8};
enum InpIdx   {I1 = 0, I2, I3, I4, I5, I6, I7, I8};

#define MOTOR_ON    TRUE
#define MOTOR_OFF   FALSE

#define STRMAXLEN   80

HANDLE  fthdl;
DWORD   errCode;
char    ComPortName[STRMAXLEN];
char    LibVersion[STRMAXLEN];

BOOL32  LightOn;


//=============================================================================

/*-----------------------------------------------------------------------------
 *  SetFlashLight  
 *---------------------------------------------------------------------------*/
static INT16 SetFlashLight(INT16 cmpVal, INT16 tmVal, INT16 tmDiff) {

    if (tmDiff == 0) {
        LightOn = false;
        SetOutPwmValues(fthdl, TA_LOCAL, OUT8, DUTY_MIN);
        return tmVal;
    }

    if ((cmpVal + tmDiff) > tmVal)
        return cmpVal;

    //  set new state
    LightOn = (!LightOn);

    //  set PWM output OUT8 duty value
    SetOutPwmValues(fthdl, TA_LOCAL, OUT8, (LightOn) ? DUTY_MAX : DUTY_MIN);
    
    return tmVal;
}

/*-----------------------------------------------------------------------------
 *  WarningLight  
 *---------------------------------------------------------------------------*/
static void WarningLight(void) {

    INT16   distance;
    BOOL32  overrun;
    INT16   cmpVal, tmVal, tmFlash;

    //  pointer all Transfer Areas
    volatile TA_ARRAY *IfTransfer = GetTransferAreasArrayAddr(fthdl);

    //  pointer Transfer Area of local interface
    volatile TA *pTA = &IfTransfer->ftxTransferArea[TA_LOCAL];

    //  pointer Timer values for time control
    volatile TA_TIMER *pTimer = &pTA->timer;

    //  set Input I1 to 'Ultrasonic sensor' mode
    SetFtUniConfig(fthdl, TA_LOCAL, I1, MODE_ULTRASONIC, TRUE);

    //  init variables
    LightOn = false;
    cmpVal  = pTimer->Timer10ms;

    //  endless loop...
    while (1) {

        //  read I1 input value from Transfer Area
        GetInIOValue(fthdl, TA_LOCAL, I1, &distance, &overrun);

        //  get timer value from timer struct in Transfer Area
        tmVal = pTimer->Timer10ms;

        //  set flash time dependent on distance
        tmFlash = (distance > 20) ? 0  :            //  no intervall, always Light off
                  (distance > 15) ? 75 :            //  intervall 750 ms
                  (distance > 10) ? 50 :            //  intervall 500 ms
                  (distance > 5)  ? 25 : 10;        //  intervall 250 ms or 100 ms

        //  setting flashlight
        cmpVal = SetFlashLight(cmpVal, tmVal, tmFlash);

        Sleep(1);
    }
}

/*-----------------------------------------------------------------------------
 *  InitTAValues  
 *---------------------------------------------------------------------------*/
static void InitTAValues() {

    //  set all motors OFF with duty= 0
    for ( int mtrIdx=0 ; mtrIdx < N_MOTOR; mtrIdx++) {
        SetOutMotorValues(fthdl, TA_LOCAL, mtrIdx, 0, 0);
        SetFtMotorConfig(fthdl, TA_LOCAL, mtrIdx, MOTOR_OFF);
    }
}

/*-----------------------------------------------------------------------------
 *  CheckParameter  
 *---------------------------------------------------------------------------*/
static int CheckCOMPar(int argc, char *argv[]) {

    char    *pStr;
    int     comNo;

    if (argc >= 2) {
        if (strlen(argv[1]) > 3) {
            if ((pStr=strstr(argv[1],"COM")) != NULL) {
                sscanf(pStr+3, "%d", &comNo);
                if (comNo >= 1 && comNo <= 255)
                    return 0;
                else {
                    cout << "WarningLight.exe: invalid COM number..." << endl << endl;
                    return 1;
                }
            }
        }
    }

    cout << "WarningLight.exe: no input given..." << endl << endl;
    return 1;
}

/*-----------------------------------------------------------------------------
 *  main
 *  
 *---------------------------------------------------------------------------*/
int main(int argc, char *argv[]) {

    cout << "\nExample WarningLight.exe ..." << endl;

    //  check input paramter
    if (CheckCOMPar(argc,argv)) {
        cout << "Usage: WarningLight.exe COMxx\t(e.g. COM2 or COM32)" << endl;
        return 1;
    }

    //  get library version
    ftxGetLibVersionStr(LibVersion, STRMAXLEN);
    cout << "\nftMscLib " << LibVersion << endl;

    //  library initialization
    errCode = ftxInitLib();

    strcpy(ComPortName, argv[1]);
    cout << "\nOpen ComPort '" << ComPortName << "' ..." << endl;

    //  open COM port
    fthdl = ftxOpenComDevice(ComPortName, 38400, &errCode);

    if (errCode == FTLIB_ERR_SUCCESS) {

        cout << "Connected to ROBO TX Controller ..." << endl;

        //  init values in Transfer Area to default
        InitTAValues();

        //  starting Transfer Area
        errCode = ftxStartTransferArea(fthdl);

        if (errCode == FTLIB_ERR_SUCCESS) {

            cout << "Transfer Area was started and runs..." << endl;

            WarningLight();

            //  stop Transfer Area
            ftxStopTransferArea(fthdl);
        }

        else {
            //  error case
            cout << "Error: Transfer Area was not started !" << endl;
        }

        //  closing port
        cout << "Closing ComPort '" << ComPortName << "' ..." << endl;
        errCode = ftxCloseDevice(fthdl);
    }

    else {
        //  error case
        cout << "Error: No interface available (Port '" << ComPortName << "')" << endl;
    }

    //  close library
    ftxCloseLib();

    return 0;
}
